using System;
using System.Data;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Brian    lin</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/26/2004</CreationDate>
	///<Note>
	///    The Food and Drug Administration classifies this software as a medical device.
	///    As such, it may not be changed in any way. Modifications to this software may result 
	///    in an adulterated medical device under 21CFR820, the use of which is considered to be 
	///    a violation of US Federal Statutes.  Acquiring and implementing this software through 
	///    the Freedom of information Act requires the implementor to assume total responsibility 
	///    for the software, and become a registered manufacturer of a medical device,
	///    subject to FDA regulations.
	///</Note>
	///<summary>Provides means to execute hospital location lookup on VistA system via VistALink.</summary>

	#endregion

	public class HospitalLocation
	{
		private const string HOSPITAL_LOCATION_LOOKUP_RPC_NAME = "VBECS HOSPITAL LOCATION LOOKUP";

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3790"> 
		///		<ExpectedInput>Valid divisionCode.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataSet with hospital locations associated with division 589 retrieved from VistA.</ExpectedOutput>xpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid = "3928">
		///		<ExpectedInput>Null divisionCode parameter.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4446"> 
		///		<ExpectedInput>VistALink is not connected.</ExpectedInput>
		///		<ExpectedOutput>InvalidOperationException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves VistA Hospital Locations and their associated Internal
		/// Entry Numbers (IENs) associated with the input division code.
		/// </summary>
		/// <param name="divisionCode">Division code string.</param>
		/// <returns>Read-only <see cref="DataSet"/> containing hospital location lookup results.</returns>
		public static DataSet GetHospitalLocationsByDivision( string divisionCode )
		{
			if( divisionCode == null )
			{
				throw( new ArgumentNullException( "divisionCode" ) );
			}
			//
			// Try cache table first
			DataSet dsHospitalLocations = VistaCache.GetHospitalLocationCache(divisionCode);
			//
			if (dsHospitalLocations != null && dsHospitalLocations.Tables.Count > 0 && dsHospitalLocations.Tables[0].Rows.Count > 0)
			{
				return dsHospitalLocations;
			}
			else
			{
				string rpcResult = VistALink.GetRpcBroker().ExecuteRpc( GetHospitalLocationsByDivisionRpcRequest(divisionCode) );
				//
				return DataTransformUtility.LoadXmlStringIntoDataSet( rpcResult );
			}
		}

		/// <summary>
		/// Creates Hospital Locations lookup <see cref="RpcRequest"/> from a given search value.
		/// </summary>
		/// <param name="searchValue">DivisionCode string</param>
		/// <returns>Hospital Location lookup RpcRequest</returns>
		private static RpcRequest GetHospitalLocationsByDivisionRpcRequest( string searchValue )
		{
			RpcRequest _hospitalLocationRequest = VistALink.CreateRpcRequest( HOSPITAL_LOCATION_LOOKUP_RPC_NAME );
			_hospitalLocationRequest.Parameters.Add( new RpcStringParameter( 1, ( searchValue != null ) ? searchValue.Trim() : String.Empty ) );

			return _hospitalLocationRequest;
		}
	}
}
